/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

#ifndef _LCL_AOA_ALGO_H_
#define _LCL_AOA_ALGO_H_

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                      Includes Section
///////////////////////////////////////////////////////////////////////////////////////////////////
#include <stdint.h>
#include "EmbeddedTypes.h"
#include "lcl_utils.h"
#include "lcl_aoa.h"

/*!
 * @addtogroup GENFSK_Localization
 * @addtogroup AngleOfArrival Angle of Arrival localization feature
 * @ingroup GENFSK_Localization
 * Angle of Arrival measurement.
 * @{
 */

/*! @file
 * GENFSK Localization - Angle of Arrival Algorithm API.
 */

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Typedef Section
///////////////////////////////////////////////////////////////////////////////////////////////////
typedef float real32_t;

///////////////////////////////////////////////////////////////////////////////////////////////////
//                                Function Prototypes Section
///////////////////////////////////////////////////////////////////////////////////////////////////

/*!
 * @brief CFO estimation for directing finding. Floating-point implementation
 * This function computes the exact tone frequency based on IQ captured by AoA module
 * It is used to determine the CFO by comparing the computed tone frequency compared to
 * the theoretical value (250kHZ)
 *
 * @param capture_p    (input) Pointer on lclAoaIqCapturePerAnt_t pointer as reported by AoA module
 * @param Nseg         Sgment length in samples
 * @param nbBlkNseg    Number of blocks of Nseg samples
 * @param toneFreq    (output) Estimated tone frequency in radians

 *
 * @retval Error status
 *
 */
lclStatus_t LclAoaAlgoToneFrequencyEstimate(lclAoaIqCapturePerAnt_t capture_p[], uint32_t Nseg, uint32_t nbBlkNseg, real32_t *toneFreq);

/*!
 * @brief CFO estimation for directing finding. Fixed-point implementation
 * This function computes the exact tone frequency based on IQ captured by AoA module
 * It is used to determine the CFO by comparing the computed tone frequency compared to
 * the theoretical value (250kHZ)
 *
 * @param capture_p    (input) Pointer on lclAoaIqCapturePerAnt_t pointer as reported by AoA module
 * @param Nseg         Sgment length in samples
 * @param nbBlkNseg    Number of blocks of Nseg samples
 * @param Iout        (output) I sample output from vector auto-correlation, Q11 format
 * @param Qout        (output) Q sample output from vector auto-correlation, Q11 format
 *
 * @retval Error status
 *
 */
lclStatus_t LclAoaAlgoToneFrequencyEstimate_FP(lclAoaIqCapturePerAnt_t capture_p[], uint32_t Nseg, uint32_t nbBlkNseg, int32_t *Iout, int32_t *Qout);

/*!
 * @brief CFO compensation for directing finding. Floating-point implementation
 * This function removes the tone (which frequency is provided in parameter) to the input signal
 *
 * @param capture_p    (input / output) Pointer on lclAoaIqCapturePerAnt_t pointer as reported by AoA module
 * @param Nseg         Segment length in samples
 * @param nbBlkNseg    Number of blocks of Nseg samples
 * @param switchPeriod Antenna switching period (in samples)
 * @param startIdx     Index of the first sample from first sampling period (in samples)
 * @param toneFreq     (input) Tone frequency in radians

 *
 * @retval Error status
 *
 */
void LclAoaAlgoToneFrequencyCompensate(lclAoaIqCapturePerAnt_t capture_p[], uint32_t Nseg, uint32_t nbBlkNseg, uint32_t switchPeriod, uint32_t startIdx, real32_t *toneFreq);

/*!
 * @brief CFO compensation for directing finding. Fixed-point implementation
 * This function removes the tone (which frequency is provided in parameter) to the input signal
 *
 * @param capture_p    (input / output) Pointer on lclAoaIqCapturePerAnt_t pointer as reported by AoA module
 * @param Nseg         Segment length in samples
 * @param nbBlkNseg    Number of blocks of Nseg samples
 * @param switchPeriod Antenna switching period (in samples)
 * @param startIdx     Index of the first sample from first sampling period (in samples)
 * @param Iout        (input) I sample output from vector auto-correlation, Q11 format
 * @param Qout        (input) Q sample output from vector auto-correlation, Q11 format

 *
 * @retval Error status
 *
 */
void LclAoaAlgoToneFrequencyCompensate_FP(lclAoaIqCapturePerAnt_t capture_p[], uint32_t Nseg, uint32_t nbBlkNseg, uint32_t switchPeriod, uint32_t startIdx, int32_t *Iout, int32_t *Qout);

/*!
 * @brief AoA Matrix Pencil directing finding algorithm
 *
 * @param Iin          input IQ signal,  block of NsegxNant samples, signed fixed-point Q15
 * @param Qin          input IQ signal,  block of NsegxNant samples, signed fixed-point Q15
 * @param Nseg         segment length in samples
 * @param avg_wind_size averaging window
 * @param PhyCst       Physical constant, PhyCst  = c/2/fc/d_array; % c: LightSpeed, fc : RF frequency, d_array : distance between 2 consecutive sensors, range : 1/4,1/2,1
 * @param Nant         number of antennas, unsigned int 4 bits
 * @param fp_en        1:fixed-point, 0:floating-point, unsigned int 1 bit
 * @param QR_Nit       number of iteration used by QR algorithm , unsigned integer 6 bits
 * @param QR_eps       small value added to ill-conditioned matrix to make it well conditioned, floating-point constant of 0.01 for the moment
 * @param AngEstOut    (output) estimated angle, floating-point Q15
 * @param QIEstOut     (output) Quality Indicator, floating-point Q15
 *
 * @retval Error status
 *
 */
lclStatus_t  LclAoaAlgoMatrixPencilSeq(real32_t *Iin, real32_t *Qin, uint32_t Nseg, uint32_t avg_wind_size, uint32_t Nsrc, real32_t PhyCst, uint32_t Nant, real32_t *AngEstOut,
                                       real32_t *QIEstOut, bool_t fp_en,uint32_t QR_NIt,float QR_eps);

#endif // _LCL_AOA_ALGO_H_
